/*
*  Spectral Products (c)
*  sm245 Linux-USB Application Program  Kernel 3.13.x (Ubuntu 14.04)
*
*  Author : Ko Seok Hyun
*  REV    : 2016.  5. 25 - 2019. 3. 4 
*  rev Ver : 1.1.4 Raspberry Pi3

INSTALLING and USING:
1) install libusb-1.0.x 
   (sudo apt-get install libusb-1.0)
2) mkdir /home/(user_dir)/sm245
3) copy sm440all.tar to /home/(user_dir)/sm440
4) tair -xvf sm440all.tar
5) type ./sm440install or manually copy files :
    (1) 88-spusb-smSerise.rules to /etc/udev/rules.d directory
    (2) sm440.hex to /lib/firmware directory
    (3) check sm440 hotplugging to linux system
       ("tail -f /var/log/syslog")
    (4) connect sm440
    (5) check device (Vid:04B4  Pid:2791)
       ("lsusb")

6) gcc -o sdksm440 sdksm440.c -L. -lSPdbUSBLinux -lm -lusb-1.0
7) ./sdksm440
8) test sm440 

*/

#include <stdio.h>
#include "sm440.h"

typedef enum {false, true} bool;

int g_iAvgTime=1;
double g_intTime = 35;
bool g_bExtTrgMode = false;

void viewinfo();
short GetAndSaveData(short sChannel);
short TrgModeSetSM440(short sTrgMode, short sChannel);

main()
{
	int mode;
	int i;
	int exitmode=1;
	short sTrgMode;
	short result=0;
    unsigned long * pDataArray;
	int sChannel = 0;
	short sTotChnlNum;
	char strModel[20],strSerial[20];

	system("clear");

	while(exitmode)
	{
		viewinfo();

		scanf("%d", &mode);
		printf("\n");			

		switch (mode)
		{ 
		case 0:
			system("clear");
			viewinfo();
			printf( "%d [Find SM440]\n", mode);
			sTotChnlNum = spFindDevice();
			if(sTotChnlNum < 0)
				printf( "     Can't find Device\n");
			break;
		case 1:
			system("clear");
			viewinfo();
			printf( "%d : [Initialize SM440]\n", mode);
			result = spDeviceInitialize();
			if(result < 0)
			{
				printf("     Device Initialized Information Error : %d", result);
				break;
			}
			else
				printf("     All Device Initialized Complete");
			break;
		case 2:
			system("clear");
			viewinfo();
			printf( "%d [Get Device Information]\n", mode);

			if(sTotChnlNum > 1)
			{
				printf("Select Channel : ");
				scanf("%d",&sChannel);
				if((sChannel >= sTotChnlNum)||(sChannel < 0))
				{
					printf("Channel select error : Range 0 ~ %d",sTotChnlNum -1);
					break;
				}
			}

			result = spDevInfo(strModel,strSerial,sChannel);
			if(result < 0)
			{
				printf("     Get Device Information Error : %d", result);
				break;
			}
			printf("     Model : %s, Serial : %s\n",strModel,strSerial);
			break;
		case 3:
			system("clear");
			viewinfo();
			printf( "%d [Set Trigger Mode]\n", mode);

			if(sTotChnlNum > 1)
			{
				printf("Select Channel : ");
				scanf("%d",&sChannel);
				if((sChannel >= sTotChnlNum)||(sChannel < 0))
				{
					printf("Channel select error : Range 0 ~ %d",sTotChnlNum -1);
					break;
				}
			}
			printf( "     Trigger Mode Select(0 : Internal, 1 : External)  : ");
			scanf("%d",&sTrgMode);

			if(sTrgMode != 0&&sTrgMode != 1)
			{
				printf("     Error : Input 0 or 1");
				break;
			}

			TrgModeSetSM440(sTrgMode,sChannel);

			break;
		case 4:
			printf( "     input = %d [Set integration Time for SM440]\n", mode);
			if(sTotChnlNum > 1)
			{
				printf("Select Channel : ");
				scanf("%d",&sChannel);
				if((sChannel >= sTotChnlNum)||(sChannel < 0))
				{
					printf("     Channel select error : Range 0 ~ %d",sTotChnlNum -1);
					break;
				}
			}
			printf( "     Enter the integration time(about 1~65535 milisecond) and enter ---- ");
			scanf("%f", &g_intTime);
			if(g_intTime >=1 && g_intTime <= 65535){
				result = spSetDblIntEx(g_intTime,sChannel);
			}
			else
			{
				printf( "     Again SP Set integration time Mode  \n");
			}
			if(result == 1){
				printf( "     Integration set ... OK \n");
			}
			else
				printf( "     Something wrong... \n");

			printf( "     Enter the Time Average and enter ---- ");
			scanf("%d", &g_iAvgTime);
			printf( "     Time Average set ... OK \n");
			break;
		case 5:
			printf( "mode = %d [Read Data from SM440]\n", mode);
			if(sTotChnlNum > 1)
			{
				printf("Select Channel : ");
				scanf("%d",&sChannel);
				if((sChannel >= sTotChnlNum)||(sChannel < 0))
				{
					printf("Channel select error : Range 0 ~ %d",sTotChnlNum -1);
					break;
				}
			}
			result = GetAndSaveData(sChannel);
			printf( "Return result = %d \n", result);
			printf("save data to Ch#%d sm440_data.txt",sChannel);
			break;
		case 6:
			printf( "     mode = %d [Close and quit]\n", mode);
			spCloseDevice();
			exitmode = 0;
			break;
		}
	}
}

void viewinfo(void)
{
		printf("\n");
		printf("\n");
		printf("\n");
		printf( "     -----------------------------------------------------\n");
		printf( "        Spectral Products SM440 Linux SDK Version 1.1.4     \n");
		printf( "     -----------------------------------------------------\n");
		printf( "     0 : Find SP sm440 Device (spFindDevice()\n");
		printf( "     1 : SP SM440 Device init (spDeviceInitialize(void))  \n");
		printf( "     2 : SP SM440 Get Device Information (spDevInfo(strModel,strSerial,sChannel)\n");
		printf( "     3 : SP SM440 Set Trigger Mode (spSetTrgEx(sTrgType,sChannel);)\n");
		printf( "     4 : SP SM440 Set InitTime & TimeAverage (spSetDbIntEx(intTime,sChannel))\n");
		printf( "     5 : SP SM440 Read Data (spReadDataEx(Array,sChannel))\n");
		printf( "     6 : SP SM440 Close Device (spCloseDevice())\n");
		printf( "     -----------------------------------------------------\n");
		printf("\n");			
		printf( "            input [0~6] =     ");
}

/****************************************************************
* SM440 supports only Free Run Previous in Internal Mode
* 1. Internal Mode
*  - Free Run Previous
* SM440 supports only Falling edge in External Mode
* 2. External Mode
*  - Falling edge
*****************************************************************/
short TrgModeSetSM440(short sTrgMode, short sChannel)
{
	short sRtn;

	if(sTrgMode == 0)
	{
		short sInterMode;

		sRtn = spSetTrgEx(SP_TRIGGER_INTERNAL, sChannel); // Set Internal Trigger Mode
		if(sRtn < 0)
		{
			printf("     Error : Set Trigger Mode - %d\n",sRtn);
			return sRtn;
		}
		g_bExtTrgMode = false;
		printf("     Trigger Set....OK\n");
	}
	else if(sTrgMode == 1)
	{
		short sEdgeMode;

		sRtn = spSetTrgEx(SP_TRIGGER_EXTERNAL, sChannel); //Set External Trigger Mode
		if(sRtn < 0)
		{
			printf("Error : Set Trigger Mode - %d\n",sRtn);
			return sRtn;
		}
		g_bExtTrgMode = true;
		printf("Trigger Set....OK\n");
	}
	else
	{
		printf("     Error : Input 0 or 1\n");
		sRtn = -100;
	}

	return sRtn;
}

short GetAndSaveData(short sChannel)
{	// Get Data from the CCD
	short sRtn;
	long plArrayData[SP_CCD_PIXEL_TOSHIBA] = {0,};
	unsigned long plTmpData[SP_CCD_PIXEL_TOSHIBA];
	int i, j;

	/*******************************************************************************************************/
	/********************* Get Data from the CCD ***********************************************************/

	if(!g_bExtTrgMode) // Not External Triggering
	{
		for(i = 0 ; i < g_iAvgTime ; i++)
		{
			sRtn = spReadDataEx(plTmpData, sChannel);
			if(sRtn < 0)
			{	// There is an error in communication via USB
				return sRtn;
			}
			for(j=0 ; j<SP_CCD_PIXEL_TOSHIBA ; j++) plArrayData[j] += (long)((double)plTmpData[j]/g_iAvgTime + 0.5);
			printf("     Get Spectrum Data.....OK\n");
		}
	}
	else if(g_bExtTrgMode)//Extern Triggering
	{
		for(i = 0 ; i < g_iAvgTime ; i++)
		{
			printf("     Waiting for External Trigger....\n");
			do
			{
				sRtn = spReadDataEx(plTmpData, sChannel);
			}while(sRtn == SP_EXT_TRG_WAITING);

			if(sRtn < 0)
				return sRtn;
			for(j=0 ; j<SP_CCD_PIXEL_TOSHIBA ; j++) plArrayData[j] += (long)((double)plTmpData[j]/g_iAvgTime + 0.5);
			printf("     Get Spectrum Data.....OK\n");
		}
	}


	/*******************************************************************************************************/
	/********************* Data converion from long to float ***********************************************/
	
	int iNum = SP_CCD_PIXEL_TOSHIBA_REAL;	// The first 32 pixels are optical blank pixels. 
											// The real data is 3648 for TOSHIBA TCD1304AP.
	float fData[SP_CCD_PIXEL_TOSHIBA_REAL] = {0,};	 

	// Data converting
	for(i=0 ; i<iNum ; i++)	fData[i] = (float)plArrayData[i+32];

	/*******************************************************************************************************/
	/********************* Calibration data converion  *****************************************************/
	
	//// Spectrometer wavelength vs. pixel number Calibration Data set ////
	//// This set was given as an example.
	//// The customer has to use the proper calibration data set per each channel
	int nCalibNo=10 ;	// A number of the calibration data
	double dWaveLength[10];
	double dPixelNo[10];
	double dRCoefficient[5];
	double dCoefficient[5];
	//Random calibration sample data (This is only for TOSHIBA TCD1304AP CCD, assumed the total pixel number is 3648) 
	dWaveLength[0] =  253.7; dPixelNo[0] = 440;
	dWaveLength[1] =  313.2; dPixelNo[1] = 711;
	dWaveLength[2] =  365.0; dPixelNo[2] = 949;
	dWaveLength[3] =  404.7; dPixelNo[3] = 1124;
	dWaveLength[4] =  435.8; dPixelNo[4] = 1260;
	dWaveLength[5] =  546.1; dPixelNo[5] = 1735;
	dWaveLength[6] =  696.5; dPixelNo[6] = 2364;
	dWaveLength[7] =  763.5; dPixelNo[7] = 2638;
	dWaveLength[8] =  811.5; dPixelNo[8] = 2834;
	dWaveLength[9] =  912.3; dPixelNo[9] = 3241;

	// Polynomial fitting
	spPolyFit(dPixelNo, dWaveLength, nCalibNo, dCoefficient, 3);
	spPolyFit(dWaveLength, dPixelNo, nCalibNo, dRCoefficient, 3);
	/*******************************************************************************************************/
	/********************* Save data  **********************************************************************/

	double dWL;

	FILE *f;
	char fname[50];
	char strSave[4000];

	sprintf(fname,"%s%d%s","Ch#",sChannel," sm440_data.txt");
	f = fopen (fname,"w"); /* open a file for writing */
	printf( "\nfopen..........\n");

	if (f == NULL) {
		printf ("\nCan not open file!\n"); /* error open file */
		return;
	}

	sprintf(strSave,"Pixel\t Wavelength(nm)\t Intensity \r\n");
	fputs(strSave,f);
	
	for(i=0 ; i<3648 ; i++){
		spPolyCalc(dCoefficient, 3, (i+1) ,&dWL);
		sprintf(strSave, " %5d\t %5.2f\t %12.4f\r\n",i+1, dWL, fData[i]);
		fputs(strSave,f);
	}
		
	fclose(f);
	return 1;
}

