/*
*  Spectral Products (c)
*  sm304 Linux-USB Application Program  Raspberry Pi3
*
*  Author : Ko Seok Hyun
*  REV    : 2016.  5. 25 - 2019. 3. 4 
*  rev Ver : 1.1.4 Raspberry Pi3

INSTALLING and USING:
1) install libusb-1.0.x 
   (sudo apt-get install libusb-1.0)
2) mkdir /home/(user_dir)/SM304
3) copy SDK_SM304.tar to /home/(user_dir)/SM304
4) tair -xvf SDK_SM304.tar
5) type ./SM304install or manually copy files :
    (1) 88-spusb-smSerise.rules to /etc/udev/rules.d directory
    (2) hex file to /lib/firmware directory
	    - SM304_G9204-512.hex,SM304_G9206-256.hex,SM304_G9206-512.hex
		  SM304_G9208-256.hex,SM304_G9208-512.hex
    (3) check SM304 hotplugging to linux system
       ("tail -f /var/log/syslog")
    (4) connect SM304
    (5) check device (VID:0X04B4 PID:0X0242)
       ("lsusb")

6) gcc -o sdksm304 sdksm304.c -L. -lSPdbUSBLinux -lm -lusb-1.0
7) ./sdksm304
8) test SM304 

*/

#include <stdio.h>
#include "sm304.h"

typedef enum {false, true} bool;

int g_iAvgTime=1;
int g_intTime = 35;
bool g_bExtTrgMode[] = {false,};
short sTotPixelNum[10];
short sRealPixelNum[10];

void viewinfo();
short GetAndSaveData(short sChannel);
short TrgModeSetSM304(short sTrgMode, short sChannel);
short ShutterControl(short sPos,short sChannel);
short TECoolerControl(long lTEC,short sChannel);
short CapacityValue(long lCF,short sChannel);
void PixelSelect(short sTotalCh);

main()
{
	int mode;
	int i;
	int exitmode=1;

	short sTrgMode;
	long lTEOnOff = 1;
	long lCFValue = 1;
	short result=0;
	short sShutterPos;
    unsigned long * pDataArray;
	int sChannel = 0;
	short sTotChnlNum;
	char strModel[20],strSerial[20];
	
	system("clear");
	
	while(exitmode)
	{
		viewinfo();

		scanf("%d", &mode);
		printf("\n");			

		switch (mode)
		{ 
		case 0:
			system("clear");
			viewinfo();
			printf( "%d [Find SM304]\n", mode);

			sTotChnlNum = spFindDevice();
			if(sTotChnlNum < 0)
				printf( "     Can't find Device\n");

			break;
		case 1:
			system("clear");
			viewinfo();
			printf( "%d : [Initialize SM304]\n", mode);
			result = spDeviceInitialize();
			if(result < 0)
			{
				printf("     Device Initialized Information Error : %d", result);
				break;
			}
			else
				printf("     All Device Initialized Complete");

			PixelSelect(sTotChnlNum);
			break;
		case 2:
			system("clear");
			viewinfo();
			printf( "%d [Get Device Information]\n", mode);

			if(sTotChnlNum > 1)
			{
				printf("Select Channel : ");
				scanf("%d",&sChannel);
				if((sChannel >= sTotChnlNum)||(sChannel < 0))
				{
					printf("Channel select error : Range 0 ~ %d",sTotChnlNum -1);
					break;
				}
			}

			result = spDevInfo(strModel,strSerial,sChannel);
			if(result < 0)
			{
				printf("     Get Device Information Error : %d", result);
				break;
			}
			printf("     Model : %s, Serial : %s\n",strModel,strSerial);
			break;
		case 3:
			system("clear");
			viewinfo();
			printf( "%d [Set Trigger Mode]\n", mode);

			if(sTotChnlNum > 1)
			{
				printf("Select Channel : ");
				scanf("%d",&sChannel);
				if((sChannel >= sTotChnlNum)||(sChannel < 0))
				{
					printf("Channel select error : Range 0 ~ %d",sTotChnlNum -1);
					break;
				}
			}
			printf( "     Trigger Mode Select(0 : Internal, 1 : External)  : ");
			scanf("%d",&sTrgMode);

			if(sTrgMode != 0&&sTrgMode != 1)
			{
				printf("     Error : Input 0 or 1");
				break;
			}

			TrgModeSetSM304(sTrgMode,sChannel);

			break;
		case 4:
			printf( "     input = %d [Set integration Time for SM304]\n", mode);
			if(sTotChnlNum > 1)
			{
				printf("Select Channel : ");
				scanf("%d",&sChannel);
				if((sChannel >= sTotChnlNum)||(sChannel < 0))
				{
					printf("Channel select error : Range 0 ~ %d",sTotChnlNum -1);
					break;
				}
			}
			printf( "     Enter the integration time(about 1~65535 milisecond) and enter ---- ");
			scanf("%d", &g_intTime);
			if(g_intTime >=1 && g_intTime <= 65535){
				result = spSetDblIntEx((double)g_intTime,sChannel);
			}
			else
			{
				printf( "     Again SP Set integration time Mode  \n");
			}
			if(result == 1){
				printf( "     Integration set ... OK \n");
			}
			else
				printf( "     Something wrong... \n");

			printf( "     Enter the Time Average and enter ---- ");
			scanf("%d", &g_iAvgTime);
			printf( "     Time Average set ... OK \n");
			break;
		case 5:
			printf( "mode = %d [Read Data from SM304]\n", mode);
			if(sTotChnlNum > 1)
			{
				printf("Select Channel : ");
				scanf("%d",&sChannel);
				if((sChannel >= sTotChnlNum)||(sChannel < 0))
				{
					printf("Channel select error : Range 0 ~ %d",sTotChnlNum -1);
					break;
				}
			}
			result = GetAndSaveData(sChannel);
			printf( "Return result = %d \n", result);
			printf("save data to Ch#%d sm304_data.txt",sChannel);
			break;
		case 6:
			system("clear");
			viewinfo();
			printf( "%d [Set Shutter Position]\n", mode);

			if(sTotChnlNum > 1)
			{
				printf("Select Channel : ");
				scanf("%d",&sChannel);
				if((sChannel >= sTotChnlNum)||(sChannel < 0))
				{
					printf("Channel select error : Range 0 ~ %d",sTotChnlNum -1);
					break;
				}
			}

			printf( "     Shutter Position Select(0 : Open, 1 : Close)  : ");
			scanf("%d",&sShutterPos);

			ShutterControl(sShutterPos,sChannel);

			break;
		case 7:
			system("clear");
			viewinfo();
			printf( "%d [Set TE-Cooler and Capacity value]\n", mode);

			if(sTotChnlNum > 1)
			{
				printf("Select Channel : ");
				scanf("%d",&sChannel);
				if((sChannel >= sTotChnlNum)||(sChannel < 0))
				{
					printf("Channel select error : Range 0 ~ %d",sTotChnlNum -1);
					break;
				}
			}

			printf( "     Set TE-Cooler Power(0 : OFF, 1 : ON)  : ");
			scanf("%d",&lTEOnOff);

			TECoolerControl(lTEOnOff,sChannel);

			printf( "     Set Capacity Value(0 : 1uF, 1 : 10uF)  : ");
			scanf("%d",&lCFValue);

			CapacityValue(lCFValue,sChannel);
		
			break;
		case 8:
			printf( "mode = %d [Close and quit]\n", mode);
			spCloseDevice();
			exitmode = 0;
			break;
		}
	}
}


void viewinfo(void)
{
		printf("\n");
		printf("\n");
		printf("\n");
		printf( "     -----------------------------------------------------\n");
		printf( "        Spectral Products SM304 Linux SDK Version 1.1.4     \n");
		printf( "     -----------------------------------------------------\n");
		printf( "     0 : Find SP SM304 Device (spFindDevice()\n");
		printf( "     1 : SP SM304 Device init (spDeviceInitialize(void))  \n");
		printf( "     2 : SP SM304 Get Device Information (spDevInfo(strModel,strSerial,sChannel)\n");
		printf( "     3 : SP SM304 Set Trigger Mode (spSetTrgEx(sTrgType,sChannel);)\n");
		printf( "     4 : SP SM304 Set InitTime & TimeAverage (spSetDblIntEx(intTime,sChannel))\n");
		printf( "     5 : SP SM304 Read Data (spReadDataEx(Array,sChannel))\n");
		printf( "     6 : SP SM304 Shutter Control (spSetShutterPos(sShutter,sChannel))\n");
		printf( "     7 : SP SM304 TE-Cooler Control (spSetTec(lTEC,sChannel))\n");
		printf( "         SP SM304 Set Capacity Value (spSelectCF(lCF,sChannel))\n");
		printf( "     8 : SP SM304 Close Device (spCloseDevice())\n");
		printf( "     -----------------------------------------------------\n");
		printf("\n");			
		printf( "            input [0~8] =     ");
}

void PixelSelect(short sTotalCh)
{
	int j;
	short sRtn;

	for(j = 0 ; j < sTotalCh; j++)
	{
 		sRtn = spGetModel(j);
		if(sRtn == SP_CCD_G9212) 
		{
			sTotPixelNum[j] = SP_CCD_PIXEL_G9212;
			sRealPixelNum[j] = SP_CCD_PIXEL_G9212_REAL;
		}
		else if(sRtn == SP_CCD_G92XX_256)
		{
			sTotPixelNum[j] = SP_CCD_PIXEL_G92XX_256;
			sRealPixelNum[j] = SP_CCD_PIXEL_G92XX_256_REAL;
		}
	}
}

/****************************************************************
* SM304 supports only Free Run Previous in Internal Mode
* 1. Internal Mode
*  - Free Run Previous
* SM304 supports only Falling edge in External Mode
* 2. External Mode
*  - Falling edge
*****************************************************************/
short TrgModeSetSM304(short sTrgMode, short sChannel)
{
	short sRtn;

	if(sTrgMode == 0)
	{
		short sInterMode;

		sRtn = spSetTrgEx(SP_TRIGGER_INTERNAL, sChannel); // Set Internal Trigger Mode
		if(sRtn < 0)
		{
			printf("     Error : Set Trigger Mode - %d\n",sRtn);
			return sRtn;
		}

		g_bExtTrgMode[sChannel] = false;
		printf("     Set Internal Trigger Mode.....OK\n");
	}
	else if(sTrgMode == 1)
	{
		short sEdgeMode;

		sRtn = spSetTrgEx(SP_TRIGGER_EXTERNAL, sChannel); //Set External Trigger Mode
		if(sRtn < 0)
		{
			printf("Error : Set Trigger Mode - %d\n",sRtn);
			return sRtn;
		}

		g_bExtTrgMode[sChannel] = true;
		printf("     Set External Trigger Mode.....OK\n");
	}
	else
	{
		printf("     Error : Input 0 or 1\n");
		sRtn = -100;
	}

	return sRtn;
}

short GetAndSaveData(short sChannel)
{	// Get Data from the CCD
	short sRtn;

	long plArrayData[1000] = {0,};
	unsigned long plTmpData[1000];
	int i, j;

	/*******************************************************************************************************/
	/********************* Get Data from the CCD ***********************************************************/

	if(!g_bExtTrgMode[sChannel]) // Not External Triggering
	{
		for(i = 0 ; i < g_iAvgTime ; i++)
		{
			sRtn = spReadDataEx(plTmpData, sChannel);
			if(sRtn < 0)
			{	// There is an error in communication via USB
				return sRtn;
			}
			for(j=0 ; j<sTotPixelNum[sChannel] ; j++) plArrayData[j] += (long)((double)plTmpData[j]/g_iAvgTime + 0.5);
			printf("     Get Spectrum Data.....OK\n");
		}
	}
	else if(g_bExtTrgMode[sChannel])// External Triggering
	{
		for(i = 0 ; i < g_iAvgTime ; i++)
		{
			printf("     Waiting for External Trigger....\n");
			do
			{
				sRtn = spReadDataEx(plTmpData, sChannel);
			}while(sRtn == SP_EXT_TRG_WAITING);

			if(sRtn < 0)
				return sRtn;

			for(j=0 ; j<sTotPixelNum[sChannel] ; j++) plArrayData[j] += (long)((double)plTmpData[j]/g_iAvgTime + 0.5);
			printf("     Get Spectrum Data.....OK\n");
		}
	}

	/*******************************************************************************************************/
	/********************* Data converion from long to float ***********************************************/
	
	int iNum = sRealPixelNum[sChannel];	
	float fData[1000] = {0,};	 

	// Data converting
	for(i=0 ; i<iNum ; i++)	fData[i] = (float)plArrayData[i+10];

	/*******************************************************************************************************/
	/********************* Calibration data converion  *****************************************************/
	//// Spectrometer wavelength vs. pixel number Calibration Data set ////
	//// This set was given as an example.
	//// The customer has to use the proper calibration data set per each channel
	int nCalibNo=13 ;	// A number of the calibration data
	double dWaveLength[13];
	double dPixelNo[13];
	double dRCoefficient[5];
	double dCoefficient[5];
	//Random calibration sample data (This is only for Hamamatsu G9212 InGaAs array detector, the total pixel number is 512) 
	dWaveLength[0] =  842.5; dPixelNo[0] = 34;
	dWaveLength[1] =  912.3; dPixelNo[1] = 68;
	dWaveLength[2] =  965.8; dPixelNo[2] = 94;
	dWaveLength[3] =  1047.0; dPixelNo[3] = 134;
	dWaveLength[4] =  1067.4; dPixelNo[4] = 144;
	dWaveLength[5] =  1148.8; dPixelNo[5] = 182;
	dWaveLength[6] =  1295.7; dPixelNo[6] = 252;
	dWaveLength[7] =  1350.4; dPixelNo[7] = 278;
	dWaveLength[8] =  1371.9; dPixelNo[8] = 287;
	dWaveLength[9] =  1409.4; dPixelNo[9] = 305;
	dWaveLength[10] =  1504.7; dPixelNo[10] = 349;
	dWaveLength[11] =  1685.0; dPixelNo[11] = 431;
	dWaveLength[12] =  1824.6; dPixelNo[12] = 494;

	// Polynomial fitting
	spPolyFit(dPixelNo, dWaveLength, nCalibNo, dCoefficient, 3);
	spPolyFit(dWaveLength, dPixelNo, nCalibNo, dRCoefficient, 3);
	/*******************************************************************************************************/
	/********************* Save data  **********************************************************************/

	double dWL;

	FILE *f;
	char fname[50];
	char strSave[4000];

	sprintf(fname,"%s%d%s","Ch#",sChannel," sm304_data.txt");
	f = fopen (fname,"w"); /* open a file for writing */
	printf( "\nfopen..........\n");

	if (f == NULL) {
		printf ("\nCan not open file!\n"); /* error open file */
		return;
	}

	sprintf(strSave,"Pixel\t Wavelength(nm)\t Intensity \r\n");
	fputs(strSave,f);
	
	for(i=0 ; i<sRealPixelNum[sChannel] ; i++){
		spPolyCalc(dCoefficient, 3, (i+1) ,&dWL);
		sprintf(strSave, " %5d\t %5.2f\t %12.4f\r\n",i+1, dWL, fData[i]);
		fputs(strSave,f);
	}
		
	fclose(f);
	return 1;
}

short ShutterControl(short sPos,short sChannel)
{
	short sRtn;
	short sCurPos;

	sRtn = spInsShutter(sChannel);
	if(sRtn != 1)
	{
		printf("     This device does not have a built-in shutter. - %d",sRtn);
		return -1;
	}

	sRtn = spGetShutterPos(sChannel);
	if(sRtn < 0)
	{
		printf("     Error : Get Shutter Position - %d\n",sRtn);
		return sRtn;
	}

	printf("     Current Shutter Position : %d\n", sRtn);

	sRtn = spSetShutterPos(sPos,sChannel);
	if(sRtn < 0)
	{
		printf("     Error : Set Shutter Position - %d\n",sRtn);
		return sRtn;
	}

	printf("     Set Shutter Position.....OK\n");
	return sRtn;
}

short TECoolerControl(long lTEC,short sChannel)
{
	short sRtn;

	sRtn = spSetTec(lTEC,sChannel);
	if(sRtn < 0)
	{
		printf("     ERROR : Set TE-Cooler Power. - %d",sRtn);
		return sRtn;
	}

	printf("     Set TE-Cooler.....OK");
	return sRtn;
}

short CapacityValue(long lCF,short sChannel)
{
	short sRtn; 

	sRtn = spSelectCF(lCF,sChannel);
	if(sRtn < 0)
	{
		printf("     ERROR : Set Capacity Value. - %d",sRtn);
		return sRtn;
	}

	printf("     Set Capacity Value.....OK");
	return sRtn;

}