/*
*  Spectral Products (c)
*  SM245 Linux-USB Application Program  Raspberry Pi3
*
*  Author : Ko Seok Hyun
*  REV    : 2016.  5. 25 - 2019. 3. 3 
*  rev Ver : 1.1.4 Raspberry Pi3

INSTALLING and USING:
1) install libusb-1.0.x 
   (sudo apt-get install libusb-1.0)
2) mkdir /home/(user_dir)/SM245
3) copy SDK_SM245.tar to /home/(user_dir)/sm245
4) tair -xvf SDK_SM245.tar
5) type ./sm245install or manually copy files :
    (1) 88-spusb-smSerise.rules to /etc/udev/rules.d directory
    (2) SM245.hex to /lib/firmware directory
    (3) check SM245 hotplugging to linux system
       ("tail -f /var/log/syslog")
    (4) connect SM245
    (5) check device (Vid:04B4  Pid:0237)
       ("lsusb")

6) gcc -o sdksm245 sdksm245.c -L. -lSPdbUSBLinux -lm -lusb-1.0
7) ./sdksm245
8) test SM245 
*/

#include <stdio.h>
#include "sm245.h"

typedef enum {false, true} bool;

int g_iAvgTime=1;
int g_intTime = 35;
bool g_bExtTrgMode = false;

void viewinfo();
short GetAndSaveData(short sChannel);
short TrgModeSetSM245(short sTrgMode, short sChannel);

main()
{
	int mode;
	int i;
	int exitmode=1;
	short sTrgMode;
	short result=0;
    unsigned long * pDataArray;
	int sChannel = 0;
	short sTotChnlNum;
	char strModel[20],strSerial[20];
	
	system("clear");
	
	while(exitmode)
	{
		viewinfo();

		scanf("%d", &mode);
		printf("\n");			

		switch (mode)
		{ 
		case 0:
			system("clear");
			viewinfo();
			printf( "%d [Find SM245]\n", mode);
			sTotChnlNum = spFindDevice();
			if(sTotChnlNum < 0)
				printf( "     Can't find Device\n");
			break;
		case 1:
			system("clear");
			viewinfo();
			printf( "%d : [Initialize SM245]\n", mode);
			result = spDeviceInitialize();
			if(result < 0)
			{
				printf("     Device Initialized Information Error : %d", result);
				break;
			}
			else
				printf("     All Device Initialized Complete");
			break;
		case 2:
			system("clear");
			viewinfo();
			printf( "%d [Get Device Information]\n", mode);

			if(sTotChnlNum > 1)
			{
				printf("Select Channel : ");
				scanf("%d",&sChannel);
				if((sChannel >= sTotChnlNum)||(sChannel < 0))
				{
					printf("Channel select error : Range 0 ~ %d",sTotChnlNum -1);
					break;
				}
			}

			result = spDevInfo(strModel,strSerial,sChannel);
			if(result < 0)
			{
				printf("     Get Device Information Error : %d", result);
				break;
			}
			printf("     Model : %s, Serial : %s\n",strModel,strSerial);
			break;
		case 3:
			system("clear");
			viewinfo();
			printf( "%d [Set Trigger Mode]\n", mode);

			if(sTotChnlNum > 1)
			{
				printf("Select Channel : ");
				scanf("%d",&sChannel);
				if((sChannel >= sTotChnlNum)||(sChannel < 0))
				{
					printf("Channel select error : Range 0 ~ %d",sTotChnlNum -1);
					break;
				}
			}
			printf( "     Trigger Mode Select(0 : Internal, 1 : External)  : ");
			scanf("%d",&sTrgMode);

			if(sTrgMode != 0&&sTrgMode != 1)
			{
				printf("     Error : Input 0 or 1");
				break;
			}

			TrgModeSetSM245(sTrgMode,sChannel);

			break;
		case 4:
			printf( "     input = %d [Set integration Time for SM245]\n", mode);
			if(sTotChnlNum > 1)
			{
				printf("Select Channel : ");
				scanf("%d",&sChannel);
				if((sChannel >= sTotChnlNum)||(sChannel < 0))
				{
					printf("     Channel select error : Range 0 ~ %d",sTotChnlNum -1);
					break;
				}
			}
			printf( "     Enter the integration time(about 1~65535 milisecond) and enter ---- ");
			scanf("%d", &g_intTime);
			if(g_intTime >=1 && g_intTime <= 65535){
				result = spSetDblIntEx((double)g_intTime,sChannel);
			}
			else
			{
				printf( "     Again SP Set integration time Mode  \n");
			}
			if(result == 1){
				printf( "     Integration set ... OK \n");
			}
			else
				printf( "     Something wrong... \n");

			printf( "     Enter the Time Average and enter ---- ");
			scanf("%d", &g_iAvgTime);
			printf( "     Time Average set ... OK \n");
			break;
		case 5:
			printf( "mode = %d [Read Data from SM245]\n", mode);
			if(sTotChnlNum > 1)
			{
				printf("Select Channel : ");
				scanf("%d",&sChannel);
				if((sChannel >= sTotChnlNum)||(sChannel < 0))
				{
					printf("     Channel select error : Range 0 ~ %d",sTotChnlNum -1);
					break;
				}
			}
			result = GetAndSaveData(sChannel);
			printf( "Return result = %d \n", result);
			printf("save data to Ch#%d sm245_data.txt",sChannel);
			break;
		case 6:
			printf( "     mode = %d [Close and quit]\n", mode);
			spCloseDevice();
			exitmode = 0;
			break;
		
		}
	}
}

void viewinfo(void)
{
		printf("\n");
		printf("\n");
		printf("\n");
		printf( "     -----------------------------------------------------\n");
		printf( "        Spectral Products SM245 Linux SDK Version 1.1.4     \n");
		printf( "     -----------------------------------------------------\n");
		printf( "     0 : Find SP sm245 Device (spFindDevice()\n");
		printf( "     1 : SP SM245 Device init (spDeviceInitialize(void))  \n");
		printf( "     2 : SP SM245 Get Device Information (spDevInfo(strModel,strSerial,sChannel)\n");
		printf( "     3 : SP SM245 Set Trigger Mode (spSetTrgEx(sTrgType,sChannel);)\n");
		printf( "     4 : SP SM245 Set InitTime & TimeAverage (spSetDbIntEx(intTime,sChannel))\n");
		printf( "     5 : SP SM245 Read Data (spReadDataEx(Array,sChannel))\n");
		printf( "     6 : SP SM245 Close Device (spCloseDevice())\n");
		printf( "     -----------------------------------------------------\n");
		printf("\n");			
		printf( "            input [0~6] =     ");
}
/****************************************************************
* SM245 dose not support Free Run Next(Internal Trigger Mode)
* 1. Internal Mode
*  - S/W Trigger Mode
*  - Free Run Previous
* 2. External Mode
*  - Falling edge
*  - Rising edge
*****************************************************************/
short TrgModeSetSM245(short sTrgMode, short sChannel)
{
	short sRtn;

	if(sTrgMode == 0)
	{
		short sInterMode;

		sRtn = spSetTrgEx(SP_TRIGGER_INTERNAL, sChannel); // Set Internal Trigger Mode
		if(sRtn < 0)
		{
			printf("     Error : Set Trigger Mode - %d\n",sRtn);
			return sRtn;
		}
		printf("     Trigger Set....OK\n");
		printf("     Set Internal Trigger Mode(0 : Software Trigger, 1 : Free Run) : ");
		scanf("%d",&sInterMode);

		//Set Internal Mode
		sRtn = spSetIntMode(sInterMode,(double)g_intTime,sChannel);
		if(sRtn < 0)
		{
			printf("     Error : Set Internal Trigger Mode - %d\n",sRtn);
			return sRtn;
		}
		g_bExtTrgMode = false;
		printf("     Set Internal Trigger Mode.....OK\n");
	}
	else if(sTrgMode == 1)
	{
		short sEdgeMode;

		sRtn = spSetTrgEx(SP_TRIGGER_EXTERNAL, sChannel); //Set External Trigger Mode
		if(sRtn < 0)
		{
			printf("Error : Set Trigger Mode - %d\n",sRtn);
			return sRtn;
		}
		
		printf("Trigger Set....OK\n");
		printf("Set External Edge Mode(0 : Falling , 1 : Rising) : ");
		scanf("%d",&sEdgeMode);

		sRtn = spSetExtEdgeMode(sEdgeMode,sChannel);
		if(sRtn < 0)
		{
			printf("Error : Set Edge Mode - %d\n",sRtn);
			return sRtn;
		}
		g_bExtTrgMode = true;
		printf("     Set Edge Mode.....OK\n");
	}
	else
	{
		printf("     Error : Input 0 or 1\n");
		sRtn = -100;
	}
		
	return sRtn;
}

short GetAndSaveData(short sChannel)
{	// Get Data from the CCD
	short sRtn;
	long plArrayData[SP_CCD_PIXEL_SONY] = {0,};
	unsigned long plTmpData[SP_CCD_PIXEL_SONY];
	int i, j;

	/*******************************************************************************************************/
	/********************* Get Data from the CCD ***********************************************************/

	if(!g_bExtTrgMode) // Not External Triggering
	{
		for(i = 0 ; i < g_iAvgTime ; i++)
		{
			sRtn = spReadDataEx(plTmpData, sChannel);

			if(sRtn < 0)
			{	// There is an error in communication via USB
				return sRtn;
			}
			for(j=0 ; j<SP_CCD_PIXEL_SONY ; j++) plArrayData[j] += (long)((double)plTmpData[j]/g_iAvgTime + 0.5);
			printf("     Get Spectrum Data.....OK\n");
		}
	}
	else if(g_bExtTrgMode) //Extern Triggering
	{
		for(i = 0 ; i < g_iAvgTime ; i++)
		{
			printf("     Waiting for External Trigger....\n");
			do
			{
				sRtn = spReadDataEx(plTmpData, sChannel);
			}while(sRtn == SP_EXT_TRG_WAITING);

			if(sRtn < 0)
				return sRtn;
			for(j=0 ; j<SP_CCD_PIXEL_SONY ; j++) plArrayData[j] += (long)((double)plTmpData[j]/g_iAvgTime + 0.5);
			printf("     Get Spectrum Data.....OK\n");
		}
	}

	/*******************************************************************************************************/
	/********************* Data converion from long to float ***********************************************/

	int iNum = SP_CCD_PIXEL_SONY_REAL;	// The first 32 pixels are optical blank pixels. 
										// The real data is 2048 for Sony ILX 511.
	float fData[SP_CCD_PIXEL_SONY_REAL] = {0,};

	// Data converting
	for(i=0 ; i<iNum ; i++)	fData[i] = (float)plArrayData[i+32];

	/*******************************************************************************************************/
	/********************* Calibration data converion  *****************************************************/
	
	//// Spectrometer wavelength vs. pixel number Calibration Data set ////
	//// This set was given as an example.
	//// The customer has to use the proper calibration data set per each channel
	int nCalibNo=12 ;	// A number of the calibration data
	double dWaveLength[12];
	double dPixelNo[12];
	double dRCoefficient[5];
	double dCoefficient[5];
	//Random calibration sample data (This is only for Sony ILX 511 CCD, assumed the total pixel number is 2048) 
	dWaveLength[0] =  253.7; dPixelNo[0] = 348;
	dWaveLength[1] =  313.2; dPixelNo[1] = 465;
	dWaveLength[2] =  365.0; dPixelNo[2] = 567;
	dWaveLength[3] =  404.7; dPixelNo[3] = 643;
	dWaveLength[4] =  435.8; dPixelNo[4] = 703;
	dWaveLength[5] =  546.1; dPixelNo[5] = 910;
	dWaveLength[6] =  577.0; dPixelNo[6] = 967;
	dWaveLength[7] =  579.1; dPixelNo[7] = 971;
	dWaveLength[8] =  696.5; dPixelNo[8] = 1186;
	dWaveLength[9] =  763.5; dPixelNo[9] = 1306;
	dWaveLength[10] =  811.5; dPixelNo[10] = 1392;
	dWaveLength[11] =  912.3; dPixelNo[11] = 1570;

	// Polynomial fitting
	spPolyFit(dPixelNo, dWaveLength, nCalibNo, dCoefficient, 3);
	spPolyFit(dWaveLength, dPixelNo, nCalibNo, dRCoefficient, 3);
	/*******************************************************************************************************/
	/********************* Save data  **********************************************************************/

	double dWL;

	FILE *f;
	char fname[50];
	char strSave[4000];

	sprintf(fname,"%s%d%s","Ch#",sChannel," sm245_data.txt");
	f = fopen (fname,"w"); /* open a file for writing */
	printf( "\nfopen..........\n");

	if (f == NULL) {
		printf ("\nCan not open file!\n"); /* error open file */
		return;
	}

	sprintf(strSave,"Pixel\t Wavelength(nm)\t Intensity \r\n");
	fputs(strSave,f);
	
	for(i=0 ; i<SP_CCD_PIXEL_SONY_REAL ; i++){
		spPolyCalc(dCoefficient, 3, (i+1) ,&dWL);
		sprintf(strSave, " %5d\t %5.2f\t %12.4f\r\n",i+1, dWL, fData[i]);
		fputs(strSave,f);
	}
		
	fclose(f);
	return 1;
}

