/*
*  Spectral Products (c)
*  sm303 Linux-USB Application Program  Raspberry Pi3
*
*  Author : Ko Seok Hyun
*  REV    : 2016.  5. 25 - 2019. 3. 4 
*  rev Ver : 1.1.4 Raspberry Pi3

INSTALLING and USING:
1) install libusb-1.0.x 
   (sudo apt-get install libusb-1.0)
2) mkdir /home/(user_dir)/SM303
3) copy SDK_SM303.tar to /home/(user_dir)/SM303
4) tair -xvf SDK_SM303.tar
5) type ./SM303install or manually copy files :
    (1) 88-spusb-smSerise.rules to /etc/udev/rules.d directory
    (2) SM303Si.hex to /lib/firmware directory
    (3) check SM303 hotplugging to linux system
       ("tail -f /var/log/syslog")
    (4) connect SM303
    (5) check device (VID:0X0547 PID:0X0322)
       ("lsusb")

6) gcc -o sdksm303 sdksm303.c -L. -lSPdbUSBLinux -lm -lusb-1.0
7) ./sdksm303
8) test SM303 

*/

#include <stdio.h>
#include "sm303.h"

typedef enum {false, true} bool;

int g_iAvgTime=1;
int g_intTime = 35;
bool g_bExtTrgMode = false;


void viewinfo();
short GetAndSaveData(short sChannel);
short TrgModeSetSM303(short sTrgMode, short sChannel);
short ShutterControl(short sPos,short sChannel);
short TECoolerControl(long lTEC,short sChannel);

main()
{
	int mode;
	int i;
	int exitmode=1;
	short sTrgMode;
	long lTEOnOff = 1;
	short result=0;
	short sShutterPos;
    unsigned long * pDataArray;
	int sChannel = 0;
	short sTotChnlNum;
	char strModel[20],strSerial[20];
	
	system("clear");
	
	while(exitmode)
	{
		viewinfo();

		scanf("%d", &mode);
		printf("\n");			

		switch (mode)
		{ 
		case 0:
			system("clear");
			viewinfo();
			printf( "%d [Find SM303]\n", mode);
			sTotChnlNum = spFindDevice();
			if(sTotChnlNum < 0)
				printf( "     Can't find Device\n");
			break;
		case 1:
			system("clear");
			viewinfo();
			printf( "%d : [Initialize SM303]\n", mode);
			result = spDeviceInitialize();
			if(result < 0)
			{
				printf("     Device Initialized Information Error : %d", result);
				break;
			}
			else
				printf("     All Device Initialized Complete");
			break;
		case 2:
			system("clear");
			viewinfo();
			printf( "%d [Get Device Information]\n", mode);

			if(sTotChnlNum > 1)
			{
				printf("Select Channel : ");
				scanf("%d",&sChannel);
				if((sChannel >= sTotChnlNum)||(sChannel < 0))
				{
					printf("Channel select error : Range 0 ~ %d",sTotChnlNum -1);
					break;
				}
			}

			result = spDevInfo(strModel,strSerial,sChannel);
			if(result < 0)
			{
				printf("     Get Device Information Error : %d", result);
				break;
			}
			printf("     Model : %s, Serial : %s\n",strModel,strSerial);
			break;
		case 3:
			system("clear");
			viewinfo();
			printf( "%d [Set Trigger Mode]\n", mode);

			if(sTotChnlNum > 1)
			{
				printf("Select Channel : ");
				scanf("%d",&sChannel);
				if((sChannel >= sTotChnlNum)||(sChannel < 0))
				{
					printf("Channel select error : Range 0 ~ %d",sTotChnlNum -1);
					break;
				}
			}
			printf( "     Trigger Mode Select(0 : Internal, 1 : External)  : ");
			scanf("%d",&sTrgMode);

			if(sTrgMode != 0&&sTrgMode != 1)
			{
				printf("     Error : Input 0 or 1");
				break;
			}

			TrgModeSetSM303(sTrgMode,sChannel);

			break;
		case 4:
			printf( "     input = %d [Set integration Time for SM303]\n", mode);
			if(sTotChnlNum > 1)
			{
				printf("Select Channel : ");
				scanf("%d",&sChannel);
				if((sChannel >= sTotChnlNum)||(sChannel < 0))
				{
					printf("Channel select error : Range 0 ~ %d",sTotChnlNum -1);
					break;
				}
			}
			printf( "     Enter the integration time(about 1~65535 milisecond) and enter ---- ");
			scanf("%d", &g_intTime);
			if(g_intTime >=1 && g_intTime <= 65535){
				result = spSetDblIntEx((double)g_intTime,sChannel);
			}
			else
			{
				printf( "     Again SP Set integration time Mode  \n");
			}
			if(result == 1){
				printf( "     Integration set ... OK \n");
			}
			else
				printf( "     Something wrong... \n");

			printf( "     Enter the Time Average and enter ---- ");
			scanf("%d", &g_iAvgTime);
			printf( "     Time Average set ... OK \n");
			break;
		case 5:
			printf( "mode = %d [Read Data from SM303]\n", mode);
			if(sTotChnlNum > 1)
			{
				printf("Select Channel : ");
				scanf("%d",&sChannel);
				if((sChannel >= sTotChnlNum)||(sChannel < 0))
				{
					printf("Channel select error : Range 0 ~ %d",sTotChnlNum -1);
					break;
				}
			}
			result = GetAndSaveData(sChannel);
			printf( "Return result = %d \n", result);
			printf("save data to Ch#%d sm303_data.txt",sChannel);
			break;
		case 6:
			system("clear");
			viewinfo();
			printf( "%d [Set Shutter Position]\n", mode);

			if(sTotChnlNum > 1)
			{
				printf("Select Channel : ");
				scanf("%d",&sChannel);
				if((sChannel >= sTotChnlNum)||(sChannel < 0))
				{
					printf("Channel select error : Range 0 ~ %d",sTotChnlNum -1);
					break;
				}
			}

			printf( "     Shutter Position Select(0 : Open, 1 : Close)  : ");
			scanf("%d",&sShutterPos);

			ShutterControl(sShutterPos,sChannel);

			break;
		case 7:
			system("clear");
			viewinfo();
			printf( "%d [Set TE-Cooler]\n", mode);

			if(sTotChnlNum > 1)
			{
				printf("Select Channel : ");
				scanf("%d",&sChannel);
				if((sChannel >= sTotChnlNum)||(sChannel < 0))
				{
					printf("Channel select error : Range 0 ~ %d",sTotChnlNum -1);
					break;
				}
			}

			printf( "     Set TE-Cooler Power(0 : OFF, 1 : ON)  : ");
			scanf("%d",&lTEOnOff);

			TECoolerControl(lTEOnOff,sChannel);
			break;
		case 8:
			printf( "mode = %d [Close and quit]\n", mode);
			spCloseDevice();
			exitmode = 0;
			break;
		}
	}
}

void viewinfo(void)
{
		printf("\n");
		printf("\n");
		printf("\n");
		printf( "     -----------------------------------------------------\n");
		printf( "        Spectral Products SM303 Linux SDK Version 1.1.4     \n");
		printf( "     -----------------------------------------------------\n");
		printf( "     0 : Find SP SM303 Device (spFindDevice()\n");
		printf( "     1 : SP SM303 Device init (spDeviceInitialize(void))  \n");
		printf( "     2 : SP SM303 Get Device Information (spDevInfo(strModel,strSerial,sChannel)\n");
		printf( "     3 : SP SM303 Set Trigger Mode (spSetTrgEx(sTrgType,sChannel);)\n");
		printf( "     4 : SP SM303 Set InitTime & TimeAverage (spSetDblIntEx(intTime,sChannel))\n");
		printf( "     5 : SP SM303 Read Data (spReadDataEx(Array,sChannel))\n");
		printf( "     6 : SP SM303 Shutter Control (spSetShutterPos(sShutter,sChannel))\n");
		printf( "     7 : SP SM303 TE-Cooler Control (spSetTec(lTEC,sChannel))\n");
		printf( "     8 : SP SM303 Close Device (spCloseDevice())\n");
		printf( "     -----------------------------------------------------\n");
		printf("\n");			
		printf( "            input [0~8] =     ");
}

/****************************************************************
* SM303 supports all mode
* 1. Internal Mode
*  - S/W Trigger Mode
*  - Free Run Previous
*  - Free Run Next
* 2. External Mode
*  - Falling edge
*  - Rising edge
*****************************************************************/
short TrgModeSetSM303(short sTrgMode, short sChannel)
{
	short sRtn;

	if(sTrgMode == 0)
	{
		short sInterMode;

		sRtn = spSetTrgEx(SP_TRIGGER_INTERNAL, sChannel); // Set Internal Trigger Mode
		if(sRtn < 0)
		{
			printf("     Error : Set Trigger Mode - %d\n",sRtn);
			return sRtn;
		}
		printf("     Trigger Set....OK\n");
		printf("     Set Internal Trigger Mode(0 : Software Trigger, 1 : Free Run Previous, 2 : Free Run Next) : ");
		scanf("%d",&sInterMode);

		//Set Internal Mode
		sRtn = spSetIntMode(sInterMode,(double)g_intTime,sChannel);
		if(sRtn < 0)
		{
			printf("     Error : Set Internal Trigger Mode - %d",sRtn);
			return sRtn;
		}
		g_bExtTrgMode = false;
		printf("     Set Internal Trigger Mode.....OK\n");
	}
	else if(sTrgMode == 1)
	{
		short sEdgeMode;

		sRtn = spSetTrgEx(SP_TRIGGER_EXTERNAL, sChannel); //Set External Trigger Mode
		if(sRtn < 0)
		{
			printf("Error : Set Trigger Mode - %d\n",sRtn);
			return sRtn;
		}

		printf("     Trigger Set....OK\n");
		printf("     Set External Edge Mode(0 : Falling , 1 : Rising) : ");
		scanf("%d",&sEdgeMode);

		sRtn = spSetExtEdgeMode(sEdgeMode,sChannel);
		if(sRtn < 0)
		{
			printf("Error : Set Edge Mode - %d\n",sRtn);
			return sRtn;
		}
		g_bExtTrgMode = true;
		printf("     Set Edge Mode.....OK\n");
	}
	else
	{
		printf("     Error : Input 0 or 1\n");
		sRtn = -100;
	}

	return sRtn;
}

short GetAndSaveData(short sChannel)
{	// Get Data from the CCD
	short sRtn;
	long plArrayData[SP_CCD_PIXEL_PDA] = {0,};
	unsigned long plTmpData[SP_CCD_PIXEL_PDA];
	int i, j;

	/*******************************************************************************************************/
	/********************* Get Data from the CCD ***********************************************************/

	if(!g_bExtTrgMode) // Not External Triggering
	{
		for(i = 0 ; i < g_iAvgTime ; i++)
		{
			sRtn = spReadDataEx(plTmpData, sChannel);
			if(sRtn < 0)
			{	// There is an error in communication via USB
				return sRtn;
			}
			for(j=0 ; j<SP_CCD_PIXEL_PDA ; j++) plArrayData[j] += (long)((double)plTmpData[j]/g_iAvgTime + 0.5);
			printf("     Get Spectrum Data.....OK\n");
		}
	}
	else if(g_bExtTrgMode)// External Triggering
	{
		for(i = 0 ; i < g_iAvgTime ; i++)
		{
			printf("     Waiting for External Trigger....\n");
			do
			{
				sRtn = spReadDataEx(plTmpData, sChannel);
			}while(sRtn == SP_EXT_TRG_WAITING);

			if(sRtn < 0)
				return sRtn;

			for(j=0 ; j<SP_CCD_PIXEL_PDA ; j++) plArrayData[j] += (long)((double)plTmpData[j]/g_iAvgTime + 0.5);
			printf("     Get Spectrum Data.....OK\n");
		}
	}

	/*******************************************************************************************************/
	/********************* Data converion from long to float ***********************************************/
	
	int iNum = SP_CCD_PIXEL_PDA_REAL;	// The first 10 pixels are optical blank pixels. 
										// The real data is 1024 for S7031.
	float fData[SP_CCD_PIXEL_PDA_REAL] = {0,};	 

	// Data converting
	for(i=0 ; i<iNum ; i++)	fData[i] = (float)plArrayData[i+10];

	/*******************************************************************************************************/
	/********************* Calibration data converion  *****************************************************/
	//// Spectrometer wavelength vs. pixel number Calibration Data set ////
	//// This set was given as an example.
	//// The customer has to use the proper calibration data set per each channel
	int nCalibNo=9 ;	// A number of the calibration data
	double dWaveLength[9];
	double dPixelNo[9];
	double dRCoefficient[5];
	double dCoefficient[5];
	//Random calibration sample data (This is only for S7031 CCD, assumed the total pixel number is 1024) 
	dWaveLength[0] =  313.2; dPixelNo[0] = 81;
	dWaveLength[1] =  365.0; dPixelNo[1] = 174;
	dWaveLength[2] =  404.7; dPixelNo[2] = 243;
	dWaveLength[3] =  435.8; dPixelNo[3] = 298;
	dWaveLength[4] =  546.1; dPixelNo[4] = 486;
	dWaveLength[5] =  696.5; dPixelNo[5] = 734;
	dWaveLength[6] =  763.5; dPixelNo[6] = 842;
	dWaveLength[7] =  811.5; dPixelNo[7] = 918;
	dWaveLength[8] =  842.5; dPixelNo[8] = 966;

	// Polynomial fitting
	spPolyFit(dPixelNo, dWaveLength, nCalibNo, dCoefficient, 3);
	spPolyFit(dWaveLength, dPixelNo, nCalibNo, dRCoefficient, 3);
	/*******************************************************************************************************/
	/********************* Save data  **********************************************************************/

	double dWL;

	FILE *f;
	char fname[50];
	char strSave[4000];

	sprintf(fname,"%s%d%s","Ch#",sChannel," sm303_data.txt");
	f = fopen (fname,"w"); /* open a file for writing */
	printf( "\nfopen..........\n");

	if (f == NULL) {
		printf ("\nCan not open file!\n"); /* error open file */
		return;
	}

	sprintf(strSave,"Pixel\t Wavelength(nm)\t Intensity \r\n");
	fputs(strSave,f);
	
	for(i=0 ; i<SP_CCD_PIXEL_PDA_REAL ; i++){
		spPolyCalc(dCoefficient, 3, (i+1) ,&dWL);
		sprintf(strSave, " %5d\t %5.2f\t %12.4f\r\n",i+1, dWL, fData[i]);
		fputs(strSave,f);
	}
		
	fclose(f);
	return 1;
}

short ShutterControl(short sPos,short sChannel)
{
	short sRtn;
	short sCurPos;

	sRtn = spInsShutter(sChannel);
	if(sRtn != 1)
	{
		printf("     This device does not have a built-in shutter. - %d",sRtn);
		return -1;
	}

	sRtn = spGetShutterPos(sChannel);
	if(sRtn < 0)
	{
		printf("     Error : Get Shutter Position - %d\n",sRtn);
		return sRtn;
	}

	printf("     Current Shutter Position : %d\n", sRtn);

	sRtn = spSetShutterPos(sPos,sChannel);
	if(sRtn < 0)
	{
		printf("     Error : Set Shutter Position - %d\n",sRtn);
		return sRtn;
	}

	printf("     Set Shutter Position.....OK\n");
	return sRtn;
}

short TECoolerControl(long lTEC,short sChannel)
{
	short sRtn;

	sRtn = spSetTec(lTEC,sChannel);
	if(sRtn < 0)
	{
		printf("     ERROR : Set TE-Cooler Power. - %d",sRtn);
		return sRtn;
	}

	printf("     Set TE-Cooler.....OK");
	return sRtn;
}